/* ------------------------------------------------------------------------ */
/* WCLASS.LIB - Classes for Clipper 5.2/5.3                                 */
/*                                                                          */
/* (c) 1996-2004 by Wolfgang Riedmann wolfgang@riedmann.it                  */
/*                                                                          */
/* WINCLASS.PRG contains a window class                                     */
/*                                                                          */
/* uses undocumented functions to create new classes in plain clipper code  */
/*                                                                          */
/* created 22/02/96                                                         */
/*                                                                          */
/* ------------------------------------------------------------------------ */

#include "box.ch"
#ifdef CLIP53
    #include "llibg.ch"
#endif

#define WIN_TOP       1
#define WIN_LEFT      2
#define WIN_BOTTOM    3
#define WIN_RIGHT     4
#define WIN_COLOR     5
#define WIN_BOX       6
#define WIN_TITLE     7
#define WIN_SHADOW    8
#define WIN_SCREEN    9
#define WIN_CURSOR    10
#define WIN_CURSROW   11
#define WIN_CURSCOL   12
#define WIN_SVCOLOR   13
#define WIN_ISSAVED   14

/* ----------------------------
* class function - returns an object of Window class
*/
FUNCTION Window( nTop, nLeft, nBottom, nRight, cColor, cBox, cTitle, lShadow )

    STATIC nClassHandle
    LOCAL  oWindow

    IF nClassHandle == NIL
        nClassHandle    := __ClassNew( "WINDOW", 14 )
        InitWinClass( nClassHandle )
    ENDIF
    oWindow     := __ClassIns( nClassHandle )
    // Parameter-Abprfung
    nTop        := IIf( ValType( nTop ) != "N", 0, nTop )
    nLeft       := IIf( ValType( nLeft ) != "N", 0, nLeft )
    nBottom     := IIf( ValType( nBottom ) != "N", 0, nBottom )
    nRight      := IIf( ValType( nRight ) != "N", 0, nRight )
    cColor      := IIf( ValType( cColor ) != "C", "", cColor )
    DO CASE
    CASE ValType( cBox ) == "N"
        cBox        := IIf( cBox == 2, B_DOUBLE, B_SINGLE ) + " "
    CASE ValType( cBox ) == "C"
        IF Len( cBox ) == 1
            cBox        := Replicate( cBox, 8 ) + " "
        ELSE
            cBox        := cBox + " "
        ENDIF
    OTHERWISE
        cBox        := ""
    ENDCASE
    cTitle      := IIf( ValType( cTitle ) != "C", "", cTitle )
    lShadow     := IIf( ValType( lShadow ) != "L", .F., lShadow )

    oWindow:top     := nTop
    oWindow:left    := nLeft
    oWindow:bottom  := nBottom
    oWindow:right   := nRight
    oWindow:color   := cColor
    oWindow:box     := cBox
    oWindow:title   := cTitle
    oWindow:shadow  := lShadow
    oWindow:saved   := .F.

    RETURN oWindow



/* =================================================================== */
/* Implementation of the class                                         */
/* =================================================================== */

/* ----------------------------
* Initialisation
*/
STATIC PROCEDURE InitWinClass( nHandle )

    // public instance variables
    __ClassAdd( nHandle, "top", "_WRWTOP" )
    __ClassAdd( nHandle, "_top", "_WRWTOP" )
    __ClassAdd( nHandle, "left", "_WRWLEFT" )
    __ClassAdd( nHandle, "_left", "_WRWLEFT" )
    __ClassAdd( nHandle, "bottom", "_WRWBOTTOM" )
    __ClassAdd( nHandle, "_bottom", "_WRWBOTTOM" )
    __ClassAdd( nHandle, "right", "_WRWRIGHT" )
    __ClassAdd( nHandle, "_right", "_WRWRIGHT" )
    __ClassAdd( nHandle, "color", "_WRWCOLOR" )
    __ClassAdd( nHandle, "_color", "_WRWCOLOR" )
    __ClassAdd( nHandle, "box", "_WRWBOX" )
    __ClassAdd( nHandle, "_box", "_WRWBOX" )
    __ClassAdd( nHandle, "title", "_WRWTITLE" )
    __ClassAdd( nHandle, "_title", "_WRWTITLE" )
    __ClassAdd( nHandle, "shadow", "_WRWSHADOW" )
    __ClassAdd( nHandle, "_shadow", "_WRWSHADOW" )
    __ClassAdd( nHandle, "saved", "_WRWISSAVED" )
    __ClassAdd( nHandle, "_saved", "_WRWISSAVED" )

    // hidden instance variables
    __ClassAdd( nHandle, "screen", "_WRWSCREEN" )
    __ClassAdd( nHandle, "_screen", "_WRWSCREEN" )
    __ClassAdd( nHandle, "cursor", "_WRWCURSOR" )
    __ClassAdd( nHandle, "_cursor", "_WRWCURSOR" )
    __ClassAdd( nHandle, "cursrow", "_WRWCURSROW" )
    __ClassAdd( nHandle, "_cursrow", "_WRWCURSROW" )
    __ClassAdd( nHandle, "curscol", "_WRWCURSCOL" )
    __ClassAdd( nHandle, "_curscol", "_WRWCURSCOL" )
    __ClassAdd( nHandle, "svscreen", "_WRWSVSCREEN" )
    __ClassAdd( nHandle, "_svscreen", "_WRWSVSCREEN" )

    // methods
    __ClassAdd( nHandle, "open", "_WRWOPEN" )
    __ClassAdd( nHandle, "close", "_WRWCLOSE" )

    RETURN


/* ----------------------------
* Get/Set-Functions for the instance variables
*/
FUNCTION _WRWTOP( uValue )

    IF PCount() > 0 .AND. ValType( uValue ) == "N"
        QSelf()[WIN_TOP]  := uValue
    ENDIF

    RETURN IIf( PCount() > 0, QSelf(), QSelf()[WIN_TOP] )

/* --------------------------- */
FUNCTION _WRWLEFT( uValue )

    IF PCount() > 0 .AND. ValType( uValue ) == "N"
        QSelf()[WIN_LEFT]  := uValue
    ENDIF

    RETURN IIf( PCount() > 0, QSelf(), QSelf()[WIN_LEFT] )

/* --------------------------- */
FUNCTION _WRWBOTTOM( uValue )

    IF PCount() > 0 .AND. ValType( uValue ) == "N"
        QSelf()[WIN_BOTTOM]  := uValue
    ENDIF

    RETURN IIf( PCount() > 0, QSelf(), QSelf()[WIN_BOTTOM] )

/* --------------------------- */
FUNCTION _WRWRIGHT( uValue )

    IF PCount() > 0 .AND. ValType( uValue ) == "N"
        QSelf()[WIN_RIGHT]  := uValue
    ENDIF

    RETURN IIf( PCount() > 0, QSelf(), QSelf()[WIN_RIGHT] )

/* --------------------------- */
FUNCTION _WRWCOLOR( uValue )

    IF PCount() > 0 .AND. ValType( uValue ) == "C"
        QSelf()[WIN_COLOR]  := uValue
    ENDIF

    RETURN IIf( PCount() > 0, QSelf(), QSelf()[WIN_COLOR] )

/* --------------------------- */
FUNCTION _WRWBOX( uValue )

    LOCAL self  := QSelf()

    IF PCount() > 0
        DO CASE
        CASE ValType( uValue ) == "N"
            self[WIN_BOX]  := IIf( uValue == 2, B_DOUBLE, B_SINGLE ) + " "
        CASE ValType( uValue ) == "C"
            IF Len( uValue ) == 1
                self[WIN_BOX]    := Replicate( uValue, 8 ) + " "
            ELSE
                self[WIN_BOX]    := uValue + " "
            ENDIF
        OTHERWISE
            self[WIN_BOX]    := ""
        ENDCASE
    ENDIF

    RETURN IIf( PCount() > 0, self, self[WIN_BOX] )

/* --------------------------- */
FUNCTION _WRWTITLE( uValue )

    IF PCount() > 0 .AND. ValType( uValue ) == "C"
        QSelf()[WIN_TITLE]  := uValue
    ENDIF

    RETURN IIf( PCount() > 0, QSelf(), QSelf()[WIN_TITLE] )

/* --------------------------- */
FUNCTION _WRWSHADOW( uValue )

    IF PCount() > 0 .AND. ValType( uValue ) == "L"
        QSelf()[WIN_SHADOW]  := uValue
    ENDIF

    RETURN IIf( PCount() > 0, QSelf(), QSelf()[WIN_SHADOW] )

/* --------------------------- */
FUNCTION _WRWSCREEN( uValue )

    IF PCount() > 0 .AND. ValType( uValue ) == "C"
        QSelf()[WIN_SCREEN]  := uValue
    ENDIF

    RETURN IIf( PCount() > 0, QSelf(), QSelf()[WIN_SCREEN] )

/* --------------------------- */
FUNCTION _WRWCURSOR( uValue )

    IF PCount() > 0 .AND. ValType( uValue ) == "N"
        QSelf()[WIN_CURSOR]  := uValue
    ENDIF

    RETURN IIf( PCount() > 0, QSelf(), QSelf()[WIN_CURSOR] )

/* --------------------------- */
FUNCTION _WRWCURSCOL( uValue )

    IF PCount() > 0 .AND. ValType( uValue ) == "N"
        QSelf()[WIN_CURSCOL]  := uValue
    ENDIF

    RETURN IIf( PCount() > 0, QSelf(), QSelf()[WIN_CURSCOL] )

/* --------------------------- */
FUNCTION _WRWCURSROW( uValue )

    IF PCount() > 0 .AND. ValType( uValue ) == "N"
        QSelf()[WIN_CURSROW]  := uValue
    ENDIF

    RETURN IIf( PCount() > 0, QSelf(), QSelf()[WIN_CURSROW] )

/* --------------------------- */
FUNCTION _WRWSVSCREEN( uValue )

    IF PCount() > 0 .AND. ValType( uValue ) == "C"
        QSelf()[WIN_SCREEN]  := uValue
    ENDIF

    RETURN IIf( PCount() > 0, QSelf(), QSelf()[WIN_SCREEN] )

/* --------------------------- */
FUNCTION _WRWISSAVED( uValue )

    IF PCount() > 0 .AND. ValType( uValue ) == "L"
        QSelf()[WIN_ISSAVED]  := uValue
    ENDIF

    RETURN IIf( PCount() > 0, QSelf(), QSelf()[WIN_ISSAVED] )

/* ----------------------------
* methods for te class
*/
FUNCTION _WRWOPEN()

    LOCAL nCounter, cTitle
    LOCAL nTop, nLeft, nBottom, nRight
    LOCAL self  := QSelf()

    IF self[WIN_RIGHT] <= 0
        self[WIN_RIGHT]    := MaxCol() + self[WIN_RIGHT]
    ENDIF
    IF self[WIN_BOTTOM] <= 0
        self[WIN_BOTTOM]    := MaxRow() + self[WIN_BOTTOM]
    ENDIF

    nTop    := self[WIN_TOP]
    nLeft   := self[WIN_LEFT]
    nBottom := self[WIN_BOTTOM]
    nRight  := self[WIN_RIGHT]

    // save only when called the first time
    IF ! self[WIN_ISSAVED]
        self[WIN_ISSAVED]    := .T.
        /* save the screen area */
        self[WIN_SCREEN]  := SaveScreen( nTop, nLeft, nBottom, nRight )

        /* save color, cursor position and cursor type */
        self[WIN_SVCOLOR]      := SetColor()
        self[WIN_CURSROW]      := Row()
        self[WIN_CURSCOL]      := Col()
        self[WIN_CURSOR]       := SetCursor()
    ENDIF

    /* modify the window according to the passed parameters */
    /* set the color */
    IF ! Empty( self[WIN_COLOR] )
        SetColor( self[WIN_COLOR] )
    ENDIF

    /* shadow */
    IF self[WIN_SHADOW]
        WinShadow( nTop, nLeft, nBottom - 1, nRight - 1 )
        nBottom := IIf( nBottom >= MaxRow(), nBottom, nBottom - 1 )
        nRight  := IIf( nRight  >= MaxCol(), nRight,  nRight  - 1 )
    ENDIF

    /* draw the box */
    IF ! Empty( self[WIN_BOX] )
        DispBox( nTop, nLeft, nBottom, nRight, self[WIN_BOX] )
    ELSE
        DispBox( nTop, nLeft, nBottom, nRight, Space(9) )
    ENDIF

    /* window title */
    cTitle  := self[WIN_TITLE]
    IF ! Empty( cTitle )
        nCounter := Int( ( nRight - nLeft - Len( cTitle ) ) / 2 ) + nLeft
        IF nCounter < nLeft
            DevPos( nTop, nLeft )
            DevOut( Left( cTitle, nRight - nLeft ) )
        ELSE
            DevPos( nTop, nCounter )
            DevOut( cTitle )
        ENDIF
    ENDIF

    RETURN self

/* --------------------------- */
FUNCTION _WRWCLOSE()

    LOCAL self  := QSelf()

    IF self[WIN_ISSAVED]
        /* restore the values, first the screen content */
        RestScreen( self[WIN_TOP], self[WIN_LEFT], ;
                    self[WIN_BOTTOM], self[WIN_RIGHT], ;
                    self[WIN_SCREEN] )

        /* restore color, position and cursor */
        SetColor( self[WIN_SVCOLOR] )
        SetPos( self[WIN_CURSROW], self[WIN_CURSCOL] )
        SetCursor( self[WIN_CURSOR] )
    ENDIF

    RETURN self

/* ---------------------------------------------------------------- */
/*    miscellaneous functions                                       */
/* ---------------------------------------------------------------- */

/* ------------------------------------------------------ *
* TextCenter( cText, nRow, [cColor], [nStartCol], [nEndCol] ) -> NIL
*/
FUNCTION TextCenter( cText, nRow, cColor, nStartCol, nEndCol )

    LOCAL nPos

    /* control and adjust the parameters */
    IF ValType( cText ) != "C" .OR. ValType( nRow ) != "N" .OR. nRow > MaxRow()
        RETURN NIL
    ENDIF
    IF ValType( nStartCol ) != "N" .OR. nStartCol < 0 .OR. nStartCol > MaxCol()
        nStartCol := 0
    ENDIF
    IF ValType( nEndCol ) != "N" .OR. nEndCol < 0 .OR. nEndCol > MaxCol()
        nEndCol := MaxCol()
    ENDIF
    nPos := ( nEndCol - nStartCol - Len( cText ) ) / 2
    IF nPos < nStartCol
        DevPos( nRow, nStartCol )
        DevOut( Left( cText, nEndCol - nStartCol ), cColor )
    ELSE
        DevPos( nRow, nPos )
        DevOut( cText, cColor )
    ENDIF

    RETURN NIL

/* ----------------------------------------------------- *
*  WinShadow( <nTop>, <nLeft>, <nBottom>, <nRight> ) --> NIL
*/
FUNCTION WinShadow( nTop, nLeft, nBottom, nRight )
    LOCAL nShadTop, nShadLeft, nShadBottom, nShadRight

#ifdef CLIP53
    IF ( ( SET( _SET_VIDEOMODE ) == LLG_VIDEO_TXT ) .OR.;
         ( SET( _SET_VIDEOMODE ) == NIL ) )
    ELSE
        RETURN NIL
    ENDIF
#endif

    nShadTop   := nShadBottom := Min(nBottom + 1, MaxRow())
    nShadLeft  := nLeft + 1
    nShadRight := Min(nRight + 1, MaxCol())

    RestScreen( nShadTop, nShadLeft, nShadBottom, nShadRight,;
       Transform( SaveScreen(nShadTop, nShadLeft, nShadBottom, nShadRight),;
       Replicate("X", nShadRight - nShadLeft + 1 ) ) )

    nShadTop    := nTop + 1
    nShadLeft   := nShadRight := Min(nRight + 1, MaxCol())
    nShadBottom := nBottom

    RestScreen( nShadTop, nShadLeft, nShadBottom, nShadRight,;
       Transform( SAVESCREEN(nShadTop,  nShadLeft , nShadBottom,  nShadRight),;
       Replicate("X", nShadBottom - nShadTop + 1 ) ) )

    RETURN NIL


/* --------------------------------------------- *
*  ScreenFooter( cText, cColor ) --> cOldFooter
*/
FUNCTION ScreenFooter( cText, cColor )

    LOCAL cOldFooter, nCounter, nCol, nRow, cSaveScreen

    IF ValType( cColor ) != "C"
        cColor  := "N/W"
    ENDIF

    nCol    := Col()
    nRow    := Row()
    cSaveScreen := SaveScreen( MaxRow(), 0, MaxRow(), MaxCol() )
    cOldFooter := ""
    FOR nCounter = 1 TO Len( cSaveScreen ) STEP 2
        cOldFooter += SubStr( cSaveScreen, nCounter, 1 )
	NEXT
    cSaveScreen := NIL
    DevPos( MaxRow(), 0 )
    DevOut( PadR( cText, MaxCol() + 1, " " ) , cColor )
    DevPos( nRow, nCol )

	RETURN( AllTrim( cOldFooter ) )

/* --------------------------------------- */
*  ScreenHeader( cText, cColor ) --> cOldHeader
*/
FUNCTION ScreenHeader( cText, cColor )

    LOCAL cOldHeader, nCounter, nRow, nCol, cSaveScreen

    IF ValType( cColor ) != "C"
        cColor  := "N/W"
    ENDIF

    nCol    := Col()
    nRow    := Row()
    cSaveScreen := SaveScreen( 0, 0, 0, MaxCol() )
	cOldHeader := ""
    FOR nCounter = 1 TO Len( cSaveScreen ) STEP 2
      cOldHeader += SubStr( cSaveScreen, nCounter, 1 )
	NEXT
    cSaveScreen := NIL
    DevPos( 0, 0 )
    DevOut( PadC( cText, MaxCol() + 1, " " ) , cColor )
    DevPos( nRow, nCol )

	RETURN cOldHeader

